#!/usr/bin/env python3

import os, platform, sys

SUPPORTED_MAYA_VERSIONS = [2024, 2025, 2026, 2027]

def registerPlugin(conf, envPathSep, baseDir=None):

    if baseDir is None:
        baseDir = os.path.dirname(os.path.abspath(__file__))

    pluginPath = os.path.normpath(os.path.join(baseDir, '..'))

    lines = []

    with open(conf, 'r',  encoding='utf-8') as fin:
        lines = fin.readlines()

    pluginLine = -1

    for n in range(len(lines)):
        # already registered, do nothing
        if pluginPath in lines[n]:
            return

        if u'MAYA_PLUG_IN_PATH' in lines[n]:
            pluginLine = n
            break

    if pluginLine > -1:
        lineSplit = lines[pluginLine].split('=')

        varName = lineSplit[0].strip()
        existingPaths = lineSplit[1].strip().split(envPathSep)

        # remove existing verge3d paths (include both "verge3d" and "maya_plugin")
        filteredPaths = []
        for path in existingPaths:
            if not ('verge3d' in path.lower() and 'maya_plugin' in path.lower()):
                filteredPaths.append(path)

        # no need to use \r on Windows
        lines[pluginLine] = varName + '=' + pluginPath + envPathSep + envPathSep.join(filteredPaths) + '\n'
    else:
        # handle cases with no new line character at EOF
        if len(lines) and lines[-1][-1] != '\n':
            lines[-1] = lines[-1] + '\n'
        lines.append(u'MAYA_PLUG_IN_PATH=' + pluginPath + '\n')

    with open(conf, 'w', encoding='utf-8') as f:
        f.writelines(lines)

def traverseMayaDirs(baseDir=None):
    system = platform.system()

    if system == 'Windows':

        import ctypes
        from ctypes.wintypes import MAX_PATH

        dll = ctypes.windll.shell32
        buf = ctypes.create_unicode_buffer(MAX_PATH + 1)
        if dll.SHGetSpecialFolderPathW(None, buf, 0x0005, False):
            mayaDir = os.path.join(buf.value, 'maya')
        else:
            print('Failed to resolve documents directory, falling back to Documents')
            mayaDir = os.path.expandvars(r'%USERPROFILE%\Documents\maya')

        envPathSep = ';'
    elif system == 'Darwin':
        mayaDir = os.path.expandvars(r'$HOME/Library/Preferences/Autodesk/maya')
        envPathSep = ':'
    else:
        mayaDir = os.path.expandvars(r'$HOME/maya')
        envPathSep = ':'

    if os.path.exists(mayaDir):
        for mayaVer in SUPPORTED_MAYA_VERSIONS:
            maxVerDir = os.path.join(mayaDir, str(mayaVer))
            envConf = os.path.join(maxVerDir, 'Maya.env')
            if os.path.exists(envConf):
                print('Registering plugin path for Maya {}'.format(mayaVer))
                registerPlugin(envConf, envPathSep, baseDir)


# Basic Verge3D installer plugin for Maya

def maya_useNewAPI():
    pass

def initializePlugin(mobject):
    import maya.api.OpenMaya as om
    mplugin = om.MFnPlugin(mobject, 'Soft8Soft', '1.0', 'Any')
    traverseMayaDirs(mplugin.loadPath())

def uninitializePlugin(mobject):
    pass


if __name__ == '__main__':
    traverseMayaDirs()
